﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Xml;
using System.Xml.XPath;
using RevisionAnalyser.Global;
using RevisionAnalyser.Projects;
using System.IO;
using RevisionAnalyser.Projects.RecoderProject.Tasks;
using SharpSvn;
using System.Security.Authentication;
using RecoderProject;

namespace RevisionAnalyser.Projects.RecoderProject
{
    public class RecoderProject : Project
    {
        private string _svnUrl = string.Empty;
        public string SvnUrl
        {
            get
            {
                return _svnUrl;
            }
            set
            {
                _svnUrl = value;
            }
        }

        private string _svnUsername = string.Empty;
        public string SvnUsername
        {
            get
            {
                return _svnUsername;
            }
            set
            {
                _svnUsername = value;
            }
        }

        private string _svnPassword = string.Empty;
        public string SvnPassword
        {
            get
            {
                return _svnPassword;
            }
            set
            {
                _svnPassword = value;
            }
        }

        private string _javaPath = string.Empty;
        public string JavaPath
        {
            get
            {
                return _javaPath;
            }
            set
            {
                _javaPath = value;
            }
        }

        private List<string> _jarLocations = new List<string>();
        public List<string> JarLocations
        {
            get
            {
                return _jarLocations;
            }
            set
            {
                _jarLocations = value;
            }
        }

        private List<string> _thrustedCertificates = new List<string>();
        public List<string> ThrustedCertificates
        {
            get
            {
                return _thrustedCertificates;
            }
            set
            {
                _thrustedCertificates = value;
            }
        }

        public override void FirstRun()
        {
            CreateFolders();
            CreateXmlNodes();
            OpenSettingsForm(0);
        }

        public override void Opened()
        {
            JavaPath = GlobalVariables.Instance.GetVariable("JAVA_PATH", LocateJava());
            LoadSettings();
            if (RevisionSet.RevisionCount > 0)
            {
                MainForm.OpenSolidSX(String.Format("{0}\\SolidSX2.db", GetAnalysedFolder()));
            }
            else
            {
                MainForm.OpenSolidSX(Constants.EMPTY_DB_PATH);
            }
        }

        public void LoadSettings()
        {
            try
            {
                SvnUrl = XmlDoc.SelectSingleNode("/RAProject/SvnSettings/Url").InnerText;
                SvnUsername = XmlDoc.SelectSingleNode("/RAProject/SvnSettings/Username").InnerText;
                SvnPassword = XmlDoc.SelectSingleNode("/RAProject/SvnSettings/Password").InnerText;
                JarLocations.Clear();
                foreach (XmlNode node in XmlDoc.SelectSingleNode("/RAProject/JarLocations").ChildNodes)
                {
                    JarLocations.Add(node.InnerText);
                }
                ThrustedCertificates.Clear();
                foreach (XmlNode node in XmlDoc.SelectSingleNode("/RAProject/ThrustedCertificates").ChildNodes)
                {
                    ThrustedCertificates.Add(node.InnerText);
                }
                RevisionSet.ClearRevisions();
                foreach (XmlNode node in XmlDoc.SelectSingleNode("/RAProject/SvnRevisions").ChildNodes)
                {
                    DateTime time;
                    long id = long.Parse(node.Attributes["ID"].InnerText);
                    string author = node.Attributes["Author"].InnerText;
                    DateTime.TryParse(node.Attributes["Time"].InnerText, out time);
                    string logMessage = node.Attributes["LogMessage"].InnerText;
                    RevisionSet.AddRevision(new SvnRevision(id, author, time, logMessage));
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show("Unable to load settings: " + ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        public override void InitMenu(ToolStripMenuItem menu)
        {
            // Generate Timeline... menu
            /*ToolStripMenuItem uxGenerateMenu = new ToolStripMenuItem("&Generate Timeline...");
            uxGenerateMenu.Click += new EventHandler(uxGenerateMenu_Click);
            uxGenerateMenu.ShortcutKeys = Keys.Control | Keys.T;
            uxGenerateMenu.Image = global::RecoderProject.Properties.Resources.TimelineIcon;
            menu.DropDownItems.Add(uxGenerateMenu);*/
            // Manage SVN Revisions... menu
            ToolStripMenuItem uxManageMenu = new ToolStripMenuItem("&Manage SVN Revisions...");
            uxManageMenu.Click += new EventHandler(uxManageMenu_Click);
            uxManageMenu.ShortcutKeys = Keys.Control | Keys.M;
            uxManageMenu.Image = global::RecoderProject.Properties.Resources.AnalyseIcon;
            menu.DropDownItems.Add(uxManageMenu);
            // Separator
            menu.DropDownItems.Add(new ToolStripSeparator());
            // Configure SVN... menu
            ToolStripMenuItem uxSvnMenu = new ToolStripMenuItem("Configure &SVN...");
            uxSvnMenu.Click += new EventHandler(uxSvnMenu_Click);
            // Configure Library Files... menu
            ToolStripMenuItem uxLibraryMenu = new ToolStripMenuItem("Configure &Libraries...");
            uxLibraryMenu.Click += new EventHandler(uxLibraryMenu_Click);
            menu.DropDownItems.Add(uxSvnMenu);
            menu.DropDownItems.Add(uxLibraryMenu);
        }

        public SvnClient GetSvnClient(string username, string password)
        {
            SvnClient client = new SvnClient();
            client.Authentication.Clear();
            client.Authentication.SslServerTrustHandlers += new EventHandler<SharpSvn.Security.SvnSslServerTrustEventArgs>(SslServerTrustHandlers);
            if (!username.Equals(string.Empty))
            {
                client.Authentication.DefaultCredentials = new System.Net.NetworkCredential(username, password);
            }
            return client;
        }

        public SvnClient GetSvnClient()
        {
            return GetSvnClient(SvnUsername, SvnPassword);
        }

        public RecoderAnalyser GetRecoderAnalyser()
        {
            return new RecoderAnalyser(_javaPath);
        }

        public string GetSvnFolder()
        {
            return String.Format("{0}\\SVN", this.ProjectFolder);
        }

        public string GetAnalysedFolder()
        {
            return String.Format("{0}\\Analysed", this.ProjectFolder);
        }

        public void CreateXmlNodes()
        {
            try
            {
                XmlNode root = XmlDoc.SelectSingleNode("RAProject");
                XmlElement svnSettings = XmlDoc.CreateElement("SvnSettings");
                svnSettings.AppendChild(XmlDoc.CreateElement("Url"));
                svnSettings.AppendChild(XmlDoc.CreateElement("Username"));
                svnSettings.AppendChild(XmlDoc.CreateElement("Password"));
                root.AppendChild(svnSettings);
                root.AppendChild(XmlDoc.CreateElement("JarLocations"));
                root.AppendChild(XmlDoc.CreateElement("ThrustedCertificates"));
                root.AppendChild(XmlDoc.CreateElement("SvnRevisions"));
                SaveXml();
            }
            catch (Exception ex)
            {
                MessageBox.Show("Unable to initialize project file: " + ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        public void OpenLibrarySettings()
        {
            
            OpenSettingsForm(2, true);
        }

        private void CreateFolders()
        {
            Directory.CreateDirectory(GetSvnFolder());
            Directory.CreateDirectory(GetAnalysedFolder());
        }

        private void OpenSettingsForm(int tabNumber)
        {
            OpenSettingsForm(tabNumber, false);
        }

        private void OpenSettingsForm(int tabNumber, bool hideOtherTabs)
        {
            SettingsForm frmSettings = new SettingsForm();
            if (frmSettings.ShowDialog(tabNumber, hideOtherTabs) == DialogResult.OK)
            {
                SvnUrl = frmSettings.GetUrl();
                SvnUsername = frmSettings.GetUsername();
                SvnPassword = frmSettings.GetPassword();
                JarLocations = frmSettings.GetLibraryLocations();
                SynchronizeSettings();
            }
        }

        private void OpenManageForm()
        {
            ManageForm frmManage = new ManageForm();
            if (frmManage.ShowDialog() == DialogResult.OK)
            {
            }
        }

        private void GenerateTimeline()
        {
            TasksForm frmTasks = new TasksForm(
                new List<Task>
                {
                    new TimelineTask()
                }, 
                false);
            frmTasks.ShowDialog();
        }

        private string LocateJava()
        {
            String path = Environment.GetEnvironmentVariable("path");
            String[] folders = path.Split(';');
            foreach (String folder in folders)
            {
                if (File.Exists(folder + "java.exe"))
                {
                    return folder;
                }
                else if (File.Exists(folder + "\\java.exe"))
                {
                    return folder + "\\";
                }
            }
            return string.Empty;
        }

        public void AddThrustedCertificate(string fingerprint)
        {
            ThrustedCertificates.Add(fingerprint);
            SynchronizeCertificates();
        }

        public void SynchronizeSettings()
        {
            try
            {
                XmlDoc.SelectSingleNode("/RAProject/SvnSettings/Url").InnerText = SvnUrl;
                XmlDoc.SelectSingleNode("/RAProject/SvnSettings/Username").InnerText = SvnUsername;
                XmlDoc.SelectSingleNode("/RAProject/SvnSettings/Password").InnerText = SvnPassword;
                XmlNode jarLoc = XmlDoc.SelectSingleNode("/RAProject/JarLocations");
                jarLoc.RemoveAll();
                foreach (string loc in JarLocations)
                {
                    XmlNode newLoc = XmlDoc.CreateElement("Location");
                    newLoc.InnerText = loc;
                    jarLoc.AppendChild(newLoc);
                }
                SaveXml();
            }
            catch (Exception ex)
            {
                MessageBox.Show("Unable to save settings: " + ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        public void SynchronizeCertificates()
        {
            XmlNode root = XmlDoc.SelectSingleNode("/RAProject/ThrustedCertificates");
            root.RemoveAll();
            foreach (string cert in ThrustedCertificates)
            {
                XmlNode newCert = XmlDoc.CreateElement("Certificate");
                newCert.InnerText = cert;
                root.AppendChild(newCert);
            }
            SaveXml();
        }

        public void SynchronizeSvnRevisions()
        {
            XmlNode root = XmlDoc.SelectSingleNode("/RAProject/SvnRevisions");
            root.RemoveAll();
            foreach (SvnRevision revision in RevisionSet.GetRevisionList())
            {
                XmlNode newRev = XmlDoc.CreateElement("SvnRevision");
                newRev.Attributes.Append(Functions.CreateXmlAttribute(XmlDoc, "ID", revision.ID.ToString()));
                newRev.Attributes.Append(Functions.CreateXmlAttribute(XmlDoc, "Author", revision.Author));
                newRev.Attributes.Append(Functions.CreateXmlAttribute(XmlDoc, "Time", revision.Time.ToString()));
                newRev.Attributes.Append(Functions.CreateXmlAttribute(XmlDoc, "LogMessage", revision.LogMessage));
                //newRev.Attributes.Append(Functions.CreateXmlAttribute(XmlDoc, "Analysed", revision.Analysed.ToString()));
                root.AppendChild(newRev);
            }
            SaveXml();
        }

        public void SslServerTrustHandlers(object sender, SharpSvn.Security.SvnSslServerTrustEventArgs e)
        {
            if (!ThrustedCertificates.Contains(e.Fingerprint))
            {
                if (TopMostMessageBox.Show(String.Format("In order to use this SVN server, you must accept the certificate and add it as a thrusted certificate to the project. Do you want to continue?\n\nCommon name: {0}\nIssuer: {1}\nValid from: {2}\nValid to: {3}", e.CommonName, e.Issuer, e.ValidFrom, e.ValidUntil), "SSL certificate error", MessageBoxButtons.YesNo, MessageBoxIcon.Warning) == DialogResult.Yes)
                {
                    AddThrustedCertificate(e.Fingerprint);
                    e.Save = true;
                }
                else
                {
                    e.Cancel = true;
                }
            }
            else
            {
                e.Save = true;
            }
        }

        void uxGenerateMenu_Click(object sender, EventArgs e)
        {
            if (MessageBox.Show("Do you want to generate a timeline for this project? Please note that this process can take a while.", "Generate Timeline", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
            {
                GenerateTimeline();
            }
        }

        void uxManageMenu_Click(object sender, EventArgs e)
        {
            OpenManageForm();
        }

        void uxSvnMenu_Click(object sender, EventArgs e)
        {
            OpenSettingsForm(0);
        }

        void uxLibraryMenu_Click(object sender, EventArgs e)
        {
            OpenSettingsForm(1);
        }
    }
}
